#ifndef SUDOKU_H_
#define SUDOKU_H_

#include <ostream>
#include <vector>
#include "misc.h"

/**
The heart of the program.  Stores the sudoku board and contains solving
algorithms.

Random notes:

It's important that the deduction routines don't return false positives.

I'm not going into details of exactly why and how the solution algorithms work.
I recommend reading http://www.scanraid.com/sudoku.htm

TODO: Throw exceptions from error conditions
*/
class Sudoku
{
private:
	class Unit;
public:
	/**
	A typedef for a number in a sudoku cell, to differentiate from other
	integers.
	*/
	typedef unsigned short Number;

	/**
	A mask is a set of bits.  They usually represent the possible numbers for a
	given cell or unit.  Since a sudoku cell can contain numbers 1-9, the lowest
	bit is unused.

	Another use for masks is to give the cells in a given unit where a given
	number may appear.  In this case, numbering is started from the lowest bit
	(bit zero) as per normal C++ indexing rules.
	*/
	typedef unsigned short Mask;

	/**
	A single sudoku cell.  Stores information about the current state of the
	cell as well as about the other valid states.  A cell is tied to
	*/
	class Cell
	{
	public:
		Cell(unsigned short, unsigned short);
		// Functions for getting various properties of the cell
		unsigned short get_x() const      { return x; }
		unsigned short get_y() const      { return y; }
		Unit     &get_row() const         { return *row; }
		Unit     &get_column() const      { return *col; }
		Unit     &get_box() const         { return *box; }
		Number   get_num() const          { return num; }
		Mask     get_mask() const         { return mask&soft_mask; }
		bool     get_mask(Number n) const { return mask&soft_mask&(1<<n); }
		bool     get_fixed() const        { return fixed; }

		// Functions for manipulating the soft mask
		void     mask_num(unsigned n)     { soft_mask&=~(1<<n); }
		void     mask_nums(Mask m)        { soft_mask&=~m; }
		void     unmask_num(unsigned n)   { soft_mask|=1<<n; }
		void     clear_soft_mask()        { soft_mask=0x3FE; }

		void     set_units(Unit &, Unit &, Unit &);
		void     set_fixed(bool f)        { fixed=f; }
		void     update_mask();
		bool     part_of(const Unit *u)   { return u==row || u==col || u==box; }
		Cell     &operator=(Number);
	private:
		unsigned short x,y;
		Number	num;
		Mask     mask;      /// Which numbers are possible by the rules
		Mask     soft_mask; /// Used for logical deduction
		bool     fixed;     /// Whether this is a given cell
		// Must use pointers because std::vector won't eat refs
		Unit     *row;
		Unit     *col;
		Unit     *box;
	};

	Sudoku();

	/**
	Sets verbosity.  If verbosity is enabled, solution steps are printed to
	stderr.
	*/
	void       set_verbose(bool v) { verbose=v; }

	void       set_cell(unsigned, unsigned, Number);
	const Cell *get_cell(unsigned, unsigned) const;
	bool       get_solved() const	{ return solved; }
	void       solve(bool =false);
	int        eliminate();
	void       load_from_string(const std::string &);
	void       fix_cells();
	void       clear_nonfixed();
private:
	/**
	A group of nine sudoku cells.  Either a row, a column or a 3x3 box.
	*/
	class Unit
	{
	public:
		enum Type
		{
			ROW,
			COL,
			BOX
		};

		Unit(const Sudoku &s, Type t, unsigned short i): sudoku(&s), type(t), index(i), mask(0x3FE) { }
		Type           get_type() const   { return type; }
		unsigned short get_index() const  { return index; }
		void           add_cell(Cell &c)  { cells.push_back(&c); }
		Mask           get_mask() const   { return mask; }
		const std::vector<Cell *> &get_cells() const	{ return cells; }
		unsigned       get_count(Number) const;
		Mask           get_cell_mask(Number) const;
		void           mask_num(Number);
		void           unmask_num(Number);
		void           update_cells();

		// Deduction routines
		int            check_uniques();
		int            reduce_intersections(const Unit &);
		int            find_naked_tuples();
		int            find_hidden_tuples();
		int            xwing(const Unit &);
	private:
		const Sudoku *sudoku;
		Type type;
		unsigned short index;  /// Only for printing solution steps
		std::vector<Cell *>	cells;
		Mask mask;
	};

	std::vector<Cell>	cells;
	std::vector<Unit>	rows;
	std::vector<Unit>	cols;
	std::vector<Unit>	boxes;
	std::vector<Unit *> units;
	bool              verbose;
	bool					solved;

	Cell       &cell(unsigned x, unsigned y)       { return cells[x+y*9]; }
	const Cell &cell(unsigned x, unsigned y) const { return cells[x+y*9]; }
	int        solve_cell(Cell &);
	void       clear_soft_masks();

	// Deduction routines
	int        reduce_intersections();
	int        xwing();
	int        brute_force();
	int        detect_impossibilities();

	friend std::ostream &operator<<(std::ostream &, Unit::Type);
};

#endif
