#include <termios.h>
#include <iostream>
#include "interface.h"

using namespace std;

Interface::Interface(Sudoku &s):
	sudoku(s),
	highlight(0),
	expand(false)
{ }

/**
Main interface loop.
*/
void Interface::run()
{
	// Set terminal to char-at-time mode, without echo
	termios	orig;
	termios	attr;
	tcgetattr(0,&orig);
	attr=orig;
	attr.c_lflag&=~ICANON;
	attr.c_lflag&=~ECHO;
	tcsetattr(0,TCSADRAIN,&attr);

	unsigned	x=0,y=0;

	draw();
	draw_candidates(sudoku.get_cell(x,y)->get_mask());

	cout<<"\e[2;2H";
	cout.flush();
	while(1)
	{
		char c=cin.get();
		if(c=='w' && y>0)
			--y;
		else if(c=='s' && y<8)
			++y;
		else if(c=='d' && x<8)
			++x;
		else if(c=='a' && x>0)
			--x;
		else if(c>='0' && c<='9')
		{
			if(highlight>9)
				highlight=c-'0';
			else
				sudoku.set_cell(x, y, c-'0');
			draw();
		}
		else if(c==' ')
		{
			sudoku.solve();
			draw();
		}
		else if(c=='t')
		{
			sudoku.solve(true);
			draw();
		}
		else if(c=='e')
		{
			sudoku.eliminate();
			draw();
		}
		else if(c=='q')
			break;
		else if(c=='?')
			highlight=10;
		else if(c=='x')
		{
			expand=!expand;
			draw();
		}
		else if(c=='f')
		{
			sudoku.fix_cells();
			draw();
		}
		else if(c=='c')
		{
			sudoku.clear_nonfixed();
			draw();
		}

		if(expand)
			cout<<"\e["<<y*4+2+y/3<<';'<<x*4+2+x/3<<'H';
		else
		{
			cout<<"\e[14H";
			draw_candidates(sudoku.get_cell(x,y)->get_mask());
			cout<<"\e["<<y+2+y/3<<';'<<x+2+x/3<<'H';
		}

		cout.flush();
	}

	// Restore terminal settings
	tcsetattr(0,TCSADRAIN,&orig);
	// Move the cursor to below the puzzle
	if(expand)
		cout<<"\e[40H";
	else
		cout<<"\e[15H";
}

/**
Draws the puzzle.
*/
void Interface::draw()
{
	cout<<"\e[H\e[J";

	if(expand)
	{
		/* Draw in expanded mode.  Each cell will get 3x3 chars.  Cell boundaries
		will be drawn with single lines, box boundaries with double.
		Beware of voodoo expressions. */
		for(unsigned i=0; i<39; ++i)
		{
			if((i%13)%4)
			{
				for(unsigned j=0; j<39; ++j)
				{
					if((j%13)%4)
					{
						unsigned k=((i%13)%4-1)*3+(j%13)%4;
						const Sudoku::Cell &c=*sudoku.get_cell((j-1-j/13)/4, (i-1-i/13)/4);
						if(c.get_num())
							cout<<c.get_num();
						else if(c.get_mask(k) && !c.get_num())
							cout<<k;
						else
							cout<<' ';
					}
					else
						cout<<'|';
				}
				cout<<'\n';
			}
			else
				cout<<"---------------------------------------\n";
		}
	}
	else
	{
		for(unsigned i=0; i<13; ++i)
		{
			if(i%4)
			{
				for(unsigned j=0; j<13; ++j)
				{
					if(j%4)
						draw_cell(*sudoku.get_cell(j-1-j/4, i-1-i/4));
					else
						cout<<'|';
				}
				cout<<'\n';
			}
			else
				cout<<"-------------\n";
		}
	}
}

/**
Visualizes the given candidate mask.  Impossible values are replaced with dots.
*/
void Interface::draw_candidates(Sudoku::Mask mask)
{
	for(Sudoku::Number i=1; i<=9; ++i)
	{
		if(!(mask&(1<<i)))
			cout<<'.';
		else
			cout<<i;
	}
}

/**
Draws a single cell in non-expanded mode.
*/
void Interface::draw_cell(const Sudoku::Cell &cell)
{
	if(highlight && !cell.get_num() && cell.get_mask(highlight))
	{
		// Highlighted cell, draw in reverse
		cout<<"\e[7m"<<highlight<<"\e[0m";
		return;
	}

	// Set color according to possible candidates
	unsigned	count=bitcount(cell.get_mask());
	if(cell.get_fixed())
		cout<<"\e[32m";
	else if(cell.get_num())
		cout<<"\e[0m";
	else if(count==3)
		cout<<"\e[36m";
	else if(count==2)
		cout<<"\e[33m";
	else if(count==1)
		cout<<"\e[35m";
	else if(count==0)
		cout<<"\e[31m";

	unsigned short	n=cell.get_num();
	if(n)
		cout<<n;
	else
		cout<<'.';

	cout<<"\e[0m";
}
